function Fval = elastica_system(t, U, U_prev, dt, params)
    m = params.m; % mass
    B = params.B; % stiffness
    ze = params.ze; % dimensionless damping coefficient
    mu = params.mu; % dimensionless friction coefficient
    g = params.g; % acceleration of gravity
    ref_t  = params.ref_t; % input velocity
    ref_tt = params.ref_tt; % input acceleration

    alpha = 0; % inclination of the sliding sleeve measured clockwise from the vertical
    
    x  = U(1); % horizontal position of the lumped mass
    y  = U(2); % vertical position of the lumped mass
    vx = U(3); % horizontal veloctiy of the lumped mass
    vy = U(4); % vertical velocity of the lumped mass
    el = U(5); % external length of the rod
    th = U(6); % deflection angle of the rod at the lumped mass
    be = U(7); % inclination of the internal force vector at the lumped mass
    vl = U(8); % time derivative of the external length of the rod

    x_prev  = U_prev(1); % values of the degrees of freedom from the previous timestep
    y_prev  = U_prev(2);
    vx_prev = U_prev(3);
    vy_prev = U_prev(4);
    el_prev = U_prev(5);
    th_prev = U_prev(6);
    be_prev = U_prev(7);
    vl_prev = U_prev(8);

    t_prev = t-dt; % time at previous timestep

    c = 2*ze*sqrt(3*m*B/el^3); % dimensional damping coefficint as function of external length of the rod
    c_prev = 2*ze*sqrt(3*m*B/el_prev^3); % previous value of the dimensional damping coefficint
    disip_x = c/m*(ref_t(t)+vx); % dissipation term in the horizontal direction
    disip_y = c/m*(vy); % dissipation term in the vertical direction
    disip_x_prev = c_prev/m*(ref_t(t_prev)+vx_prev); % previous values of the disipation terms
    disip_y_prev = c_prev/m*(vy_prev);

    k = k_param(th-1e-9,be); % k parameter of the elastica
    k2 = k^2;
    sig = sigma_0_param(k,be); % sigma parameter of the elastica

    Ki = ellipticF(sig,k2);
    Ei = ellipticE(sig,k2);
    [Kc,Ec] = ellipke(k2);

    k_prev = k_param(th_prev-1e-9,be_prev); % previous value of k parameter of the elastica
    k2_prev = k_prev^2;
    sig_prev = sigma_0_param(k_prev,be_prev); % previous value of  sigma parameter of the elastica

    Ki_prev = ellipticF(sig_prev,k2_prev);
    Ei_prev = ellipticE(sig_prev,k2_prev);
    [Kc_prev,Ec_prev] = ellipke(k2_prev);

    AA = A_param(Ki,Kc,Ei,Ec);
    BB = B_param(Ki,Kc,k,sig);

    Nx = -B/el^2*(Kc-Ki)^2*sin(be);
    Ny = -B/el^2*(Kc-Ki)^2*cos(be);

    Nx_prev = -B/el_prev^2*(Kc_prev-Ki_prev)^2*sin(be_prev);
    Ny_prev = -B/el_prev^2*(Kc_prev-Ki_prev)^2*cos(be_prev);

    f1 = - g*sin(-alpha) - ref_tt(t) - disip_x - Nx/m;
    f2 = - g*cos(-alpha) - disip_y - Ny/m;

    f1_prev = - g*sin(-alpha) - ref_tt(t_prev) - disip_x_prev - Nx_prev/m;
    f2_prev = - g*cos(-alpha) - disip_y_prev - Ny_prev/m;

    epsilon_sign = 1e-4;
    sign_vl = vl/sqrt(vl^2+epsilon_sign^2);
    abs_Nx = sqrt(Nx^2+epsilon_sign^2); % abs(Nx);%

    Fval(1) = vx - vx_prev - 0.5*dt*( f1 + f1_prev );
    Fval(2) = vy - vy_prev - 0.5*dt*( f2 + f2_prev );
    Fval(3) = dt*(x - el*( AA*sin(be) - BB*cos(be) ));
    Fval(4) = dt*(y - el*( AA*cos(be) + BB*sin(be) ));
    Fval(5) = dt*(Ny + (Ny*x-Nx*y)^2/2/B - mu*abs_Nx*sign_vl);
    Fval(6) = x - x_prev - 0.5*dt*(vx+vx_prev);
    Fval(7) = y - y_prev - 0.5*dt*(vy+vy_prev);
    Fval(8) = el - el_prev - 0.5*dt*(vl+vl_prev);
    Fval = real(Fval); % keep the real part of Fval. Imaginary components may arise due to floating point arithmetic errors
end


function k = k_param(th, be)
    k = sin(0.5*(th-be));
end

function sig = sigma_0_param(k,be)
    sig = -asin(sin(be/2)/k);
end

function A = A_param(Ki,Kc,Ei,Ec)
    A = -1 + 2*(Ec-Ei)/(Kc-Ki);
end

function B = B_param(Ki,Kc,k,sig)
    B = -2*k*cos(sig)/(Kc-Ki);
end
